/**
* \file: mlink_gst_v0.10.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* MLINK GST Adapter
*
* \component: mlink
*
* \author: Venkata Sai Ganesh Deety RBEI/ECF34 VenkataSaiGanesh.Deety@in.bosch.com
*
* \copyright: (c) 2003 - 2013 ADIT Corporation
*
* \history
* 0.1 Venkata Sai Ganesh Deety Initial version
*
***********************************************************************/

#ifndef GST_VERSION_1

#include <mlink_gst_internals.h>
#include <sys/socket.h>
#include <gst/controller/gstcontroller.h>

/* PRQA: Lint Message 826: deactivation because casting mechanism of GObject throws the finding */
/*lint -e826*/

/* PRQA: Lint Message 751, 572: deactivation because these are from GStreamer internals  */
/*lint -e751 -e572*/

static GstElement * get_sink_element (mlink_rtp_out_context * p_ctx);
static gboolean mlink_gst_volume_src_buffer_probe(GstPad *pad, GstMiniObject *obj,
        gpointer userData);

static gboolean mlink_gst_rtp_sink_buffer_probe(GstPad *pad, GstMiniObject *obj,
        gpointer userData);

gboolean mlink_gst_send_hello(mlink_rtp_out_context * p_ctx)
{
    const char data = 1;
    struct sockaddr_in address;

    address.sin_family = AF_INET;
    address.sin_addr.s_addr = htonl(p_ctx->ipaddr);
    address.sin_port = htons((uint16_t) p_ctx->port);

    sendto(p_ctx->sock, &data, sizeof(data), 0, (struct sockaddr *) &address, sizeof(address));

    return TRUE;
}

gboolean mlink_gst_send_last_rtp(mlink_rtp_in_context * p_ctx)
{
    struct sockaddr_in address;

    GstBuffer * buffer;
    const guint payload_len = 100;
    guint8 pad_len = 0;
    guint8 csrc_count = p_ctx->csrc_count;

    guint8  *payloadptr = NULL;

    buffer = gst_rtp_buffer_new_allocate (payload_len, pad_len, csrc_count);

    /* Set header for last packet */
    gst_rtp_buffer_set_marker (buffer, TRUE);
    gst_rtp_buffer_set_payload_type(buffer, p_ctx->payloadtype);
    gst_rtp_buffer_set_seq(buffer, p_ctx->sequence_number + 1);
    gst_rtp_buffer_set_timestamp(buffer, p_ctx->timestamp + 347);
    gst_rtp_buffer_set_ssrc(buffer, p_ctx->ssrc_identifier);

    /* Set audio data to 0 */
    payloadptr = (guint8*) gst_rtp_buffer_get_payload(buffer);
    memset (payloadptr, 0, payload_len);

    address.sin_family = AF_INET;
    address.sin_addr.s_addr = htonl(p_ctx->ipaddr);
    address.sin_port = htons((uint16_t) p_ctx->port);

    int result = sendto(p_ctx->sock, GST_BUFFER_DATA(buffer), GST_BUFFER_SIZE(buffer), 0, (struct sockaddr *) &address, sizeof(address));

    if (result != -1)
    {
        mlink_gst_log(p_ctx->p_dlt_ctx, DLT_LOG_INFO,"mlink_gst: Send the marker packet successful");
    }
    else
    {
        mlink_gst_log_vargs(p_ctx->p_dlt_ctx,DLT_LOG_ERROR,"mlink_gst: Send to socket failed, error: %s",strerror(errno));
    }

    g_clear_pointer (&buffer, gst_buffer_unref);

    return TRUE;
}

gboolean mlink_gst_stop_rtp_out_mainloop(gpointer data)
{
    mlink_rtp_out_context * p_ctx = (mlink_rtp_out_context *) data;
    gst_element_post_message(p_ctx->pipeline,
            gst_message_new_application(NULL, NULL));

    return FALSE;
}

gboolean mlink_gst_stop_rtp_in_mainloop(gpointer data)
{
    mlink_rtp_in_context * p_ctx = (mlink_rtp_in_context *) data;

    gst_element_post_message(p_ctx->pipeline,
            gst_message_new_application(NULL, NULL));

    return FALSE;
}

gboolean mlink_gst_run_rtp_out_buffer_probe(GstPad *pad, GstMiniObject *mini_obj, gpointer user_data)
{
    pad = pad;
    mini_obj = mini_obj;

    mlink_rtp_out_context * p_ctx = (mlink_rtp_out_context *) user_data;

    GstBuffer * buffer = GST_BUFFER(mini_obj);

    if (buffer && buffer->size == sizeof(struct rtp_header))
    {
        // Reject empty rtp packages as they will cause the buffering
        // to fail. The Galaxy SIII responds to the HELLO-message with an
        // empty rtp package
        return FALSE;
    }

    if (buffer && p_ctx->data_cb)
    {

        if (buffer->size >= sizeof(struct rtp_header))
        {
            guint32 * extension = NULL;
            guint16 extension_size = 0;
            guint8 markerbit = 0;

            struct rtp_header * p_rtp_header = (struct rtp_header *) buffer->data;

            markerbit = p_rtp_header->marker;

            if (p_rtp_header->extension)
            {
                guint8 csrc_count = p_rtp_header->csrc_count;
                guint header_extension_offset = sizeof(struct rtp_header) + csrc_count * 4;
                guint8 * ext = (buffer->data + header_extension_offset + 4);

                if (buffer->size >= header_extension_offset + 4)
                {
                    extension_size = GST_READ_UINT16_BE(buffer->data + header_extension_offset + 2);
                    if (buffer->size >= header_extension_offset + 4 + extension_size * 4)
                    {
                        extension = (guint32 *) ext;
                    }
                    else
                    {
                        extension_size = 0;
                        extension = NULL;
                    }
                }
            }

            /*
             * If marker bit is set to 1, let's save the information for the
             * buffer probe on the rtpdepay. We shouldn't send disconnection
             * request if we receive new data, or multiple close concatenated
             * messages will fail!
             */

            if (p_ctx->marker_bit_observed == TRUE && markerbit == 0)
            {
                p_ctx->new_data_after_marker_bit_observed = TRUE;
            }

            if (markerbit == 1)
            {
                p_ctx->marker_bit_observed = TRUE;

                /*
                 * If we receive the marker bit 1 for disconnection, let's in
                 * principle declare that we haven't seen new data. That means
                 * disconnection will happen, only if we receive new data will
                 * be prevented
                 */
                p_ctx->new_data_after_marker_bit_observed = FALSE;
            }


            /*
             * Always send zero from the probe on udpsrc sink as markerbit,
             * so that as soon as data is received the connection is to Audio
             * is asked
             */
            markerbit = 0;
            (p_ctx->data_cb)(p_ctx->p_user_ctx, extension, extension_size, markerbit);
        }
    }


    /*
     * Workaround for Samsung S6: re-timestamp RTP packets
     *
     * Samsung S6 timestamps RTP packets differently. When user RESUMES
     * playback after few seconds of PAUSE, the difference in rtp-timestamp
     * between the last packet after PAUSE and first packet after RESUME is
     * too high and the Low/High water mark buffering in rtpjitterbuffer does
     * not work as expected.
     *
     * This solution will work for all phones .
     */
    if(p_ctx->isFirstRTPPacket == TRUE)
    {
    	/* Save first packet RTP Timestamp and payload length */
    	p_ctx->prevRTPTimestamp = gst_rtp_buffer_get_timestamp(buffer);
    	p_ctx->prevPayloadLen = gst_rtp_buffer_get_payload_len(buffer);
    	p_ctx->isFirstRTPPacket = FALSE;
    }
    else
    {
    	/* set the current packet RTP timestamp based on previous packet payload length
    	 * currentRTPTimestamp = PreRTPTimestamp + prevPayloadLen/NumChannels/Bitdepth(in bytes)
    	 */
    	p_ctx->currRTPTimestamp =  p_ctx->prevRTPTimestamp + (p_ctx->prevPayloadLen / 2 / 2) ;
    	gst_rtp_buffer_set_timestamp(buffer, p_ctx->currRTPTimestamp);

    	/*Save the current RTP Timestamp and payload length to previous*/
    	p_ctx->prevPayloadLen = gst_rtp_buffer_get_payload_len(buffer);
    	p_ctx->prevRTPTimestamp = p_ctx->currRTPTimestamp;
    }

    return TRUE;
}

gboolean mlink_gst_run_rtp_out_add_buffer_probe(mlink_rtp_out_context * p_ctx, GstElement * element)
{
    GstIterator *it = gst_element_iterate_src_pads(element);
    gpointer p;
    gboolean retval = FALSE;

    if (it)
    {
        GstIteratorResult rv = gst_iterator_next(it, &p);
        if (rv == GST_ITERATOR_OK)
        {
            GstPad* pad = GST_PAD(p);
            if (pad)
            {
                gst_pad_add_buffer_probe(pad, (GCallback) mlink_gst_run_rtp_out_buffer_probe, p_ctx);
                g_clear_pointer (&pad, gst_object_unref);
                retval = TRUE;
            }
        }
        gst_iterator_free(it);
    }

    return retval;
}

static gboolean mlink_gst_volume_src_buffer_probe(GstPad *pad, GstMiniObject *obj,
        gpointer userData)
{
    pad = pad;
    mlink_rtp_out_context *p_ctx = (mlink_rtp_out_context *) userData;

    GstBuffer *buffer = GST_BUFFER(obj);
    GstClockTime preSinkBufferTime = GST_BUFFER_TIMESTAMP(buffer);
    p_ctx->nextPreSinkBufferTime = preSinkBufferTime
            + GST_BUFFER_DURATION(buffer);
    return TRUE;
}

static gboolean mlink_gst_rtp_sink_buffer_probe(GstPad *pad, GstMiniObject *obj,
        gpointer userData)
{
    pad = pad;
    mlink_rtp_out_context * p_ctx = (mlink_rtp_out_context *) userData;

    GstBuffer * buffer = GST_BUFFER(obj);

    if (buffer && buffer->size == sizeof(struct rtp_header))
    {
        // Reject empty rtp packages as they will cause the buffering
        // to fail. The Galaxy SIII responds to the HELLO-message with an
        // empty rtp package
        return FALSE;
    }

    if (buffer && p_ctx->data_cb)
    {
        if (buffer->size >= sizeof(struct rtp_header))
        {
            guint32 * extension = NULL;
            guint16 extension_size = 0;
            guint8 markerbit = 0;

            struct rtp_header * p_rtp_header = (struct rtp_header *) buffer->data;

            markerbit = p_rtp_header->marker;

            if (p_rtp_header->extension)
            {
                guint8 csrc_count = p_rtp_header->csrc_count;
                guint header_extension_offset = sizeof(struct rtp_header) + csrc_count * 4;
                guint8 * ext = (buffer->data + header_extension_offset + 4);

                if (buffer->size >= header_extension_offset + 4)
                {
                    extension_size = GST_READ_UINT16_BE(buffer->data + header_extension_offset + 2);
                    if (buffer->size >= header_extension_offset + 4 + extension_size * 4)
                    {
                        extension = (guint32 *) ext;
                    }
                    else
                    {
                        extension_size = 0;
                        extension = NULL;
                    }
                }
            }

            if (markerbit == 1 && p_ctx->new_data_after_marker_bit_observed == FALSE)
            {
                mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_INFO,"mlink_gst: marker bit sent via callback to application");
                usleep(30000);
                (p_ctx->data_cb)(p_ctx->p_user_ctx, extension, extension_size, markerbit);
            }
        }
    }

    return TRUE;
}

static GstElement * get_sink_element(mlink_rtp_out_context * p_ctx)
{
    gpointer item = NULL;
    GstIterator* it = gst_bin_iterate_sinks(GST_BIN(p_ctx->pipeline));
    if (GST_ITERATOR_OK != gst_iterator_next(it, &item))
    {
        item = NULL;
    }

    gpointer noitem = NULL;
    if (GST_ITERATOR_DONE != gst_iterator_next(it, &noitem))
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_INFO,"mlink_gst: More than one sink found!");
    }
    gst_iterator_free(it);

    GstElement *sinkElement = GST_ELEMENT(item);
    if (!sinkElement)
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_WARN,"mlink_gst: Did not find any sink element");
        return NULL;
    }

    return sinkElement;
}

void *mlink_gst_run_rtp_out_pipeline(void * parameter)
{
    mlink_rtp_out_context * p_ctx = (mlink_rtp_out_context *) parameter;
    gchar * caps_str = NULL;
    GSource * hello_timeout_source = NULL;

    p_ctx->g_main_context = g_main_context_new();

    g_main_context_push_thread_default(p_ctx->g_main_context);

    p_ctx->gst_mainloop = g_main_loop_new(p_ctx->g_main_context, FALSE);

    GstBus * bus;
    GstElement * src = gst_element_factory_make("udpsrc", "udpsrc");
    GstElement * rtpbin = gst_element_factory_make("gstrtpbin", "rtp_queue");

    GstElement * rtp = gst_element_factory_make("rtpL16depay", "rtpL16depay");
    GstElement * convert = gst_element_factory_make("audioconvert", "audioconvert");
    GstElement * volume = gst_element_factory_make("volume", "volume");

    GstElement * fakesink = gst_element_factory_make("fakesink", "fakesink");
    GstElement * outputSelector = gst_element_factory_make ("output-selector", "output-selector");
    GstElement * valve = gst_element_factory_make ("valve", "valve");

    GError * error = NULL;
    GstElement * sink = gst_parse_launch(p_ctx->playback_device, &error);
    if (error)
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_ERROR,"mlink_gst: Unable parse rtp out pipeline");
        g_error_free(error);
    }

    if (!src || !rtp || !rtpbin || !convert || !volume || !outputSelector || !valve || !sink || !fakesink
            || !p_ctx->pipeline || !p_ctx->gst_mainloop)
    {
        g_clear_pointer (&src, gst_object_unref);
        g_clear_pointer (&rtpbin, gst_object_unref);
        g_clear_pointer (&rtp, gst_object_unref);
        g_clear_pointer (&convert, gst_object_unref);
        g_clear_pointer (&volume, gst_object_unref);
        g_clear_pointer (&outputSelector, gst_object_unref);
        g_clear_pointer (&valve, gst_object_unref);
        g_clear_pointer (&sink, gst_object_unref);
        g_clear_pointer (&fakesink, gst_object_unref);
        g_clear_pointer (&p_ctx->pipeline, gst_object_unref);
        p_ctx->pipeline = NULL;

        if (p_ctx->gst_mainloop)
        {
            g_clear_pointer (&p_ctx->gst_mainloop, g_main_loop_unref);
        }
        if (p_ctx->error_cb)
        {
            (p_ctx->error_cb)(p_ctx->p_user_ctx, p_ctx, "could not create all elements");
        }
        return 0;
    }

    bus = gst_pipeline_get_bus(GST_PIPELINE(p_ctx->pipeline));
    gst_bus_set_sync_handler(bus, mlink_gst_rtp_out_bus_call, p_ctx);
    g_clear_pointer (&bus, gst_object_unref);

    gst_bin_add_many(GST_BIN(p_ctx->pipeline), src, rtpbin, rtp, convert, volume, outputSelector, valve, sink, NULL);

    p_ctx->volumeElement = volume;
    p_ctx->outputSelector = outputSelector;
    p_ctx->valveElement = valve;
    p_ctx->sinkElement = get_sink_element (p_ctx);
    gst_object_ref (p_ctx->sinkElement);

    gst_bin_add (GST_BIN(p_ctx->pipeline), fakesink);   /* add fakesink later to not confuse get_sink_element() */

    caps_str = g_strdup_printf("application/x-rtp, payload=%d, clock-rate=%d, channels=%d",
                                p_ctx->payloadtype, p_ctx->samplerate, p_ctx->channels);
    GstCaps * caps = gst_caps_from_string(caps_str);
    g_free(caps_str);

    gst_element_link_pads_filtered(src, NULL, rtpbin, NULL, caps);
    g_clear_pointer (&caps, gst_caps_unref);

    g_signal_connect(rtpbin, "pad-added",
            G_CALLBACK(mlink_gst_run_rtp_out_on_pad_added), rtp);

    /* buffer-mode 2: Low/High water mark buffering  */
    g_object_set(G_OBJECT(rtpbin), "buffer-mode", 2, NULL);
    g_object_set(G_OBJECT(rtpbin), "latency", p_ctx->ipl_ms, NULL);
    g_object_set(G_OBJECT(rtpbin), "ignore-pt", TRUE, NULL);


    gst_element_link_many(rtp, convert, volume, NULL);

    /* SWGIII-25884: add capsfilter between volume and output-selector */

    gchar * caps_filter = "audio/x-raw-int, endianness = 1234, signed = true, width = 16, depth = 16, channels = 2";
    caps = gst_caps_from_string(caps_filter);

    gst_element_link_pads_filtered(volume, NULL, outputSelector, NULL, caps);
    g_clear_pointer (&caps, gst_caps_unref);

    gst_element_link(valve, sink);

    p_ctx->outputSelectorPad1 = gst_element_get_request_pad (outputSelector, "src%d");
    GstPad *sinkPad = gst_element_get_static_pad (valve, "sink");
    if (!sinkPad || gst_pad_link (p_ctx->outputSelectorPad1, sinkPad) != GST_PAD_LINK_OK) {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_WARN,"mlink_gst: failed to link pads selector1");
    }
    g_clear_pointer (&sinkPad, gst_object_unref);

    GstPad *sinkRTP = gst_element_get_static_pad (rtp, "sink");
    if (sinkRTP)
    {
        gst_pad_add_buffer_probe(sinkRTP,
                (GCallback) mlink_gst_rtp_sink_buffer_probe, p_ctx);
    }

    p_ctx->outputSelectorPad2 = gst_element_get_request_pad (outputSelector, "src%d");
    GstPad *fakesinkPad = gst_element_get_static_pad (fakesink, "sink");
    if (gst_pad_link (p_ctx->outputSelectorPad2, fakesinkPad) != GST_PAD_LINK_OK) {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_WARN,"mlink_gst: failed to link pads selector2");
    }
    g_clear_pointer (&fakesinkPad, gst_object_unref);

    g_object_set (G_OBJECT (outputSelector), "pad-negotiation-mode", 2, NULL);
    g_object_set (G_OBJECT (outputSelector), "resend-latest", TRUE, NULL);
    g_object_set (G_OBJECT (outputSelector), "active-pad", p_ctx->outputSelectorPad1, NULL);
    g_object_set (G_OBJECT (fakesink), "async", FALSE, NULL);
    g_object_set (G_OBJECT (p_ctx->sinkElement), "async", FALSE, "sync", FALSE, NULL);

    mlink_gst_run_rtp_out_add_buffer_probe(p_ctx, src);
    GstPad *volumePad = gst_element_get_static_pad (volume, "src");
    p_ctx->volumeBufferProbeHandlerId = gst_pad_add_buffer_probe(volumePad,
        (GCallback) mlink_gst_volume_src_buffer_probe, p_ctx);

    gst_element_set_state(p_ctx->pipeline, GST_STATE_PLAYING);

    g_object_get(G_OBJECT(src), "sock", &p_ctx->sock, NULL);


    if (p_ctx->sock == -1)
    {
        gst_element_set_state(p_ctx->pipeline, GST_STATE_NULL);
        g_clear_pointer (&p_ctx->pipeline, gst_object_unref);
        g_clear_pointer (&p_ctx->gst_mainloop, g_main_loop_unref);
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_ERROR,"mlink_gst: could not get network socket");

        if (p_ctx->error_cb)
        {
            (p_ctx->error_cb)(p_ctx->p_user_ctx, p_ctx, "could not get network socket");
        }
        return 0;
    }

    mlink_gst_send_hello(p_ctx);

    hello_timeout_source = g_timeout_source_new(1000);
    g_source_set_callback(hello_timeout_source, (GSourceFunc) mlink_gst_send_hello, p_ctx, NULL);
    g_source_attach(hello_timeout_source, p_ctx->g_main_context);
    g_clear_pointer (&hello_timeout_source, g_source_unref);

    mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_INFO,"mlink_gst: Starting RTP out streaming... ");

    g_mutex_lock(&p_ctx->mutex);
    p_ctx->init_finished = 1;
    g_cond_signal(&p_ctx->init_cond);
    g_mutex_unlock(&p_ctx->mutex);

    g_main_loop_run(p_ctx->gst_mainloop);

    mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_INFO,"mlink_gst: Stopping RTP playback");
    gst_element_set_state(p_ctx->pipeline, GST_STATE_NULL);

    gst_pad_remove_buffer_probe (volumePad, p_ctx->volumeBufferProbeHandlerId);
    g_clear_pointer (&p_ctx->sinkElement, gst_object_unref);

    g_clear_pointer (&p_ctx->pipeline, gst_object_unref);
    g_clear_pointer (&p_ctx->gst_mainloop, g_main_loop_unref);

    g_main_context_pop_thread_default(p_ctx->g_main_context);
    g_clear_pointer (&p_ctx->g_main_context, g_main_context_unref);

    return 0;
}

void schedule_fade(GstElement* volume, GstClockTime fadeStart,
        double oldVol, double newVol, guint32 fadeMs,
        mlink_gst_ramp_type rampType, DltContext * p_dlt_ctx)
{
    GObject* volumeObject = G_OBJECT(volume);

    GstController * controller = gst_object_get_controller(volumeObject);
    GstInterpolationControlSource* csource = NULL;
    if (controller)
    {
        /* Resuse existing controller instance (from previous fade) */
        GstControlSource* controlSource = gst_controller_get_control_source(
                controller, "volume");
        csource = GST_INTERPOLATION_CONTROL_SOURCE(controlSource);
    }
    else
    {
        /* Create new controller instance */
        controller = gst_object_control_properties(volumeObject, "volume",
                NULL);
        csource = gst_interpolation_control_source_new();
        gst_controller_set_control_source(controller, "volume",
                GST_CONTROL_SOURCE(csource));
        gst_object_ref(csource);
    }

    GstInterpolateMode mode = GST_INTERPOLATE_NONE;
    switch (rampType)
    {
        case MLINK_GST_RAMP_NONE:
            mode = GST_INTERPOLATE_NONE;
            break;
        case MLINK_GST_RAMP_LINEAR:
            mode = GST_INTERPOLATE_LINEAR;
            break;
        case MLINK_GST_RAMP_CUBIC:
            mode = GST_INTERPOLATE_CUBIC;
            break;
    }
    if (!gst_interpolation_control_source_set_interpolation_mode(csource, mode))
    {
        mlink_gst_log(p_dlt_ctx,DLT_LOG_DEBUG,"mlink_gst: failed to set interpolation mode");
    }

    GValue oldValue = G_VALUE_INIT;
    g_value_init(&oldValue, G_TYPE_DOUBLE);
    g_value_set_double(&oldValue, oldVol);

    GValue newValue = G_VALUE_INIT;
    g_value_init(&newValue, G_TYPE_DOUBLE);
    g_value_set_double(&newValue, newVol);

    GstClockTime startTime = fadeStart;
    GstClockTime endTime = fadeStart + fadeMs * GST_MSECOND;

    gst_interpolation_control_source_set(csource, startTime, &oldValue);
    gst_interpolation_control_source_set(csource, endTime, &newValue);

    DLT_LOG(*p_dlt_ctx, DLT_LOG_DEBUG,
            DLT_STRING("mlink_gst: Fade start time"),
            DLT_FLOAT64((double)startTime / GST_SECOND),
            DLT_STRING("/ end time"),
            DLT_FLOAT64((double)endTime / GST_SECOND),
            DLT_STRING("/ ramp time ms"),
            DLT_UINT32(fadeMs));

    g_clear_pointer (&csource, gst_object_unref);
}
/*
 * Buffer probe attached to src pad of rtpL16pay.Modify RTP packet
 * (GstBuffer->data) to add header extension to each rtp packet sent
 * to udpsink.
 *
 * Ref:(CCC-TS-012_Audio, Chapter 5.1):Voice Command audio over RTP
 * MUST use the RTP header extension with appCategory = 0xF0000010
 * (Voice Command Engine).
 * Profile identifier for the extension header; MUST be 0x388C.
 *
 * Example RTP Payload for Voice Command:
 *
 *  0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |V=2|P|X|  CC   |M|     PT      |       sequence number         |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |                           timestamp                           |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |           synchronization source (SSRC) identifier            |
   +=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+
   |            contributing source (CSRC) identifiers             |
   |                             ....                              |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |   profile Id = 0x388C         |           length=2            |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |  Header Extension::Application Identifier  (can be zero )     |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |  Header Extension::Application Category =  0xF0000010 (for VC)|
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
   |   ........... audio payload (type = 99)..........             |
   +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
gboolean mlink_gst_run_rtp_in_buffer_probe(GstPad *pad, GstMiniObject *mini_obj, gpointer user_data)
{
    pad = pad;
    mini_obj = mini_obj;
    user_data = user_data ;

    guint16 extnprofileid = 0x388C ;    /*As per spec CCC-TS-012_Audio ch:5.1*/
    guint16 extnlen       = 2;          /*App ID (0) + App Catagory (0xF0000010)*/
    guint32 appcatagory   = 0xF0000010 ;

    guint8  *rtphdrextnptr = NULL;
    guint    rtppacketlen  = 0;
    guint    rtppayloadlen = 0;

    GstBuffer * buffer  =  GST_BUFFER(mini_obj);

    mlink_rtp_in_context * p_ctx = (mlink_rtp_in_context *) user_data;

    struct rtp_header * p_rtp_header = (struct rtp_header *) buffer->data;

    p_ctx-> sequence_number = ntohs(p_rtp_header-> sequence_number);
    p_ctx-> timestamp = ntohl(p_rtp_header-> timestamp);
    p_ctx-> ssrc_identifier = ntohl(p_rtp_header-> ssrc_identifier);
    p_ctx-> csrc_count = p_rtp_header-> csrc_count;

    rtppacketlen  = gst_rtp_buffer_get_packet_len(buffer);
    rtppayloadlen = gst_rtp_buffer_get_payload_len(buffer);

    /* Allocate additional memory of 12 bytes for Header extension */
    gst_rtp_buffer_set_packet_len(buffer, (rtppacketlen + RTPIN_VC_HEADER_EXTN_SIZE) );

    /* Get pointer to rtp audio payload where extension element would be inserted */
    rtphdrextnptr = (guint8*) gst_rtp_buffer_get_payload(buffer);
    /*move payload further 12 bytes and create 12 bytes space between RTP Hdr and Payload*/
    memmove (rtphdrextnptr + RTPIN_VC_HEADER_EXTN_SIZE, rtphdrextnptr, rtppayloadlen );
    memset (rtphdrextnptr, 0, RTPIN_VC_HEADER_EXTN_SIZE);

    /* Set extension header profile id and extension length */
    gst_rtp_buffer_set_extension_data (buffer, extnprofileid, extnlen);

    /* Set App ID and App Catagory */
    GST_WRITE_UINT32_BE ((rtphdrextnptr + 4), 0);           //app id set to 0;
    GST_WRITE_UINT32_BE ((rtphdrextnptr + 8), appcatagory); //app catagory set to 0xF0000010;

    return TRUE;
}

gboolean mlink_gst_run_rtp_in_add_buffer_probe(mlink_rtp_in_context * p_ctx, GstElement * element)
{
    GstIterator *it = gst_element_iterate_src_pads(element);
    gpointer p;
    gboolean retval = FALSE;

    if (it)
    {
        GstIteratorResult rv = gst_iterator_next(it, &p);
        if (rv == GST_ITERATOR_OK)
        {
            GstPad* pad = GST_PAD(p);
            if (pad)
            {
                gst_pad_add_buffer_probe(pad, (GCallback) mlink_gst_run_rtp_in_buffer_probe, p_ctx);
                g_clear_pointer (&pad, gst_object_unref);
                retval = TRUE;
            }
        }
        gst_iterator_free(it);
    }

    return retval;
}

void *mlink_gst_run_rtp_in_pipeline(void * parameter)
{
    mlink_rtp_in_context * p_ctx = (mlink_rtp_in_context *) parameter;
    gchar * caps_str = NULL;
    gchar * ip_str = NULL;

    p_ctx->g_main_context = g_main_context_new();

    g_main_context_push_thread_default(p_ctx->g_main_context);

    p_ctx->gst_mainloop = g_main_loop_new(p_ctx->g_main_context, FALSE);

    GstBus * bus;
    GError * error = NULL;
    GstElement * src = gst_parse_launch(p_ctx->capture_device, &error);
    GstElement * queue = gst_element_factory_make("queue", "queue");
    GstElement * audioconvert = gst_element_factory_make("audioconvert",
            "audioconvert");
    GstElement * audioresample = gst_element_factory_make("audioresample",
            "audioresample");
    GstElement * rtp = gst_element_factory_make("rtpL16pay", "rtpL16pay");

    GstElement * sink = gst_element_factory_make("udpsink", "udpsink");

    if (error)
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_ERROR,"mlink_gst: Unable parse pipeline");
        g_error_free(error);
    }
    if (!src || !rtp || !queue || !audioconvert || !audioresample || !sink
            || !p_ctx->pipeline || !p_ctx->gst_mainloop)
    {
        g_clear_pointer (&src, gst_object_unref);
        g_clear_pointer (&rtp, gst_object_unref);
        g_clear_pointer (&queue, gst_object_unref);
        g_clear_pointer (&audioconvert, gst_object_unref);
        g_clear_pointer (&audioresample, gst_object_unref);
        g_clear_pointer (&sink, gst_object_unref);
        g_clear_pointer (&p_ctx->pipeline, gst_object_unref);
        p_ctx->pipeline = NULL;

        if (p_ctx->gst_mainloop)
        {
            g_clear_pointer (&p_ctx->gst_mainloop, g_main_loop_unref);
        }
        if (p_ctx->error_cb)
        {
            (p_ctx->error_cb)(p_ctx->p_user_ctx, p_ctx,
                    "could not create all elements");
        }
        return 0;
    }

    bus = gst_pipeline_get_bus(GST_PIPELINE(p_ctx->pipeline));
    gst_bus_set_sync_handler(bus, mlink_gst_rtp_in_bus_call, p_ctx);
    g_clear_pointer (&bus, gst_object_unref);

    gst_bin_add_many(GST_BIN(p_ctx->pipeline), src, queue, audioconvert,
            audioresample, rtp, sink, NULL);

    caps_str = g_strdup_printf("audio/x-raw-int, rate=%d, channels=%d",
                                p_ctx->samplerate, p_ctx->channels);
    GstCaps * caps = gst_caps_from_string(caps_str);
    g_free(caps_str);

    /*As per specification set rtpL16depay payload type to 99 */
    g_object_set(G_OBJECT(rtp), "pt", p_ctx->payloadtype, NULL);

    gst_element_link_many(src, queue, audioresample, audioconvert, NULL);
    gst_element_link_pads_filtered(audioconvert, NULL, rtp, NULL, caps);
    gst_element_link_many(rtp, sink, NULL);
    g_clear_pointer (&caps, gst_caps_unref);

    mlink_gst_run_rtp_in_add_buffer_probe(p_ctx, rtp);

    gst_element_set_state(p_ctx->pipeline, GST_STATE_PLAYING);

    ip_str = g_strdup_printf("%d.%d.%d.%d",
                             (p_ctx->ipaddr & 0xFF000000) >> 24,
                             (p_ctx->ipaddr & 0x00FF0000) >> 16,
                             (p_ctx->ipaddr & 0x0000FF00) >> 8,
                             (p_ctx->ipaddr & 0x000000FF));

    g_object_set(G_OBJECT(sink), "port", (gint) p_ctx->port, NULL);
    g_object_set(G_OBJECT(sink), "host", ip_str, NULL);
    g_free(ip_str);
    mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_INFO,"mlink_gst: Starting RTP IN streaming...");

    g_mutex_lock(&p_ctx->mutex);
    p_ctx->init_finished = 1;
    g_cond_signal(&p_ctx->init_cond);
    g_mutex_unlock(&p_ctx->mutex);

    g_main_loop_run(p_ctx->gst_mainloop);

    GstStateChangeReturn result = gst_element_set_state(p_ctx->pipeline, GST_STATE_PAUSED);
    if (result == GST_STATE_CHANGE_ASYNC)
    {
        result = gst_element_get_state(p_ctx->pipeline, NULL, NULL, GST_CLOCK_TIME_NONE);
    }

    g_object_get(G_OBJECT(sink), "sock", &p_ctx->sock, NULL);
    if (p_ctx->sock != -1)
    {
        mlink_gst_send_last_rtp(p_ctx);
    }
    else
    {
        mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_ERROR,"mlink_gst: Could not get network socket");
    }

    gst_element_set_state(p_ctx->pipeline, GST_STATE_NULL);

    mlink_gst_log(p_ctx->p_dlt_ctx,DLT_LOG_INFO,"mlink_gst: Stopping RTP capturing");

    g_clear_pointer (&p_ctx->pipeline, gst_object_unref);
    g_clear_pointer (&p_ctx->gst_mainloop, g_main_loop_unref);

    g_main_context_pop_thread_default(p_ctx->g_main_context);
    g_clear_pointer (&p_ctx->g_main_context, g_main_context_unref);

    return 0;
}


/*lint +e751 +e572*/
/*lint +e826*/

#endif /* GST_VERSION_1 */
